#!/usr/bin/perl

#############################################################################
# Navi Diary Message Server - ndpop
# Version 1.0.1
#############################################################################

# Set perl modules path if needed
#BEGIN {unshift(@INC,'~/public_html/cgi-bin/lib/');}

use strict;
use integer;

#############################################################################
# Preferences
#############################################################################

my $logfile = '../../log/ndpop-log';
my $loglock = '../../lock/ndpop-lock';

my $usersfile = '../../navidiary/users';

my $sessionfile = '../../navidiary/session';
my $sessionlock = '../../lock/sessionlock';

my $messagefile = 'message';

my $sessionExpiredSeconds = 30;
my $challengeCodeIntervalSeconds = 10;

#############################################################################
# Global Variables
#############################################################################

my $responsePrinted = '';

#############################################################################
# Subroutines
#############################################################################

sub try (&@)
{
    my($try,$finally) = @_;
    eval { &$try };
    if ($@) {
        my $message = $@;
        &$finally;
        die $message;
    }
    else {
		&$finally;
	}
}

sub finally (&)
{
	$_[0]
}

# Prints out response lines.
sub response(@)
{
	print @_;
	$responsePrinted = 1;
}

# Finds the user in the user file by the specified user name.
sub finduser($)
{
	my ($username) = @_;
	my %result;
	open(USERS, $usersfile) || die;
	SEARCH: while(<USERS>)
	{
		chomp;
		if (!/^\#/ && /^([^:]+):([^:]+):([^:]+):([^:]+)$/)
		{
			if ($1 eq $username)
			{
				%result = (
					'username' => $1,
					'digestpassword' => $2,
					'basicpassword' => $3,
					'homedir' => $4
				);
				last SEARCH;
			}
		}
	}
	close(USERS);
	return %result;
}

# Does the digest authorization by the user name and digest.
sub digest_authorize($$)
{
	my ($username, $digest) = @_;
	my $certificated = '';
	my %userinfo = finduser($username);
	if (%userinfo)
	{
		if (make_digest($userinfo{'digestpassword'}) eq $digest)
		{
			return %userinfo;
		}
	}
	return ();
}

# Does the basic authorization by the user name and password.
sub basic_authorize($$)
{
	my ($username, $password) = @_;
	my $certificated = '';
	my %userinfo = finduser($username);
	if (%userinfo)
	{
		if ($userinfo{'basicpassword'} eq $password)
		{
			return %userinfo;
		}
	}
	return ();
}

# Checks whether the specified session key is valid.
sub verify_session($)
{
	my ($sessionKey) = @_;
	my $username = get_session_key_user($sessionKey);
	if ($username eq '')
	{
		my $challenge = get_current_challenge();
		response "-ERR 401 $challenge Invalid Session Key\n";
		exit;
	}
	my %userinfo = finduser($username);
	unless (%userinfo)
	{
		response "-ERR 400 User Not Found\n";
		exit;
	}
	return %userinfo;
}

# Creates a new session for the user which attempt to start the session.
sub create_session($)
{
	my ($paramusername) = @_;
	my $sessionKey = '';
	lock($sessionlock) || die;
	try {
		my @list = read_session_file();
		my $alreadyOpened = '';
		foreach(@list)
		{
			my ($username, $sessionKey, $timeout) = split /:/;
			if ($username eq $paramusername)
			{
				$alreadyOpened = 1;
				last;
			}
		}
		unless ($alreadyOpened)
		{
			$sessionKey = generate_session_key();
			open(SESSION, '>' . $sessionfile) || die;
			foreach(@list)
			{
				print SESSION $_;
			}
			print SESSION $paramusername . ':' . $sessionKey . ':' . (time() + $sessionExpiredSeconds) . "\n";
			close(SESSION);
		}
	}
	finally	{
		unlink($sessionlock);
	};
	return $sessionKey;
}

# Deletes the session with the specified session key.
sub delete_session($)
{
	my ($sessionKeyToDelete) = @_;
	lock($sessionlock) || die;
	try {
		my @list = read_session_file();
		open(SESSION, '>' . $sessionfile) || die;
		foreach(@list)
		{
			my ($username, $sessionKey, $timeout) = split /:/;
			print "$_\n" unless $sessionKey eq $sessionKeyToDelete;
		}
		close(SESSION);
	}
	finally {
		unlink($sessionlock);
	};
}

# Returns the user name of the owner of the specified session key.
sub get_session_key_user($)
{
	my ($sessionKeyToFind) = @_;
	my @list;
	lock($sessionlock) || die;
	try	{
		@list = read_session_file();
	}
	finally {
		unlink($sessionlock);
	};
	foreach(@list)
	{
		my ($username, $sessionKey, $timeout) = split /:/;
		if ($sessionKey eq $sessionKeyToFind)
		{
			return $username;
		}
	}
	return '';
}

# Reads the session file.
sub read_session_file()
{
	my @sessiones;
	open(SESSION, $sessionfile) || die;
	while(<SESSION>)
	{
		chomp;
		my ($username, $sessionKey, $timeout) = split /:/;
		push @sessiones, $_ if (time() < $timeout);
	}
	close(SESSION);
	return @sessiones;
}

# Generates the session key which can be used newly.
sub generate_session_key()
{
	my @base = (0..9, 'A'..'Z', 'a'..'z');
	my $key;
	for (my $i = 0; $i < 8; ++$i)
	{
		$key .= $base[rand scalar @base];
	}
	return $key;
}

# Returns the current challenge code.
sub get_current_challenge()
{
	return '<' . (time() / $challengeCodeIntervalSeconds) . '@' . $ENV{'HTTP_HOST'} . '>';
}

# Makes a digest by the specified password and the current challenge code.
sub make_digest($)
{
	my ($password) = @_;
	my $source = get_current_challenge() . $password;
	if (eval { require Digest::MD5; })
	{
		import Digest::MD5 'md5_hex';
		return md5_hex($source);
	}
	if (eval { require Digest::Perl::MD5; })
	{
		import Digest::Perl::MD5 'md5_hex';
		return md5_hex($source);
	}
	die;
}

# Acquires the lock using the file.
sub lock($)
{
	my ($lockFile) = @_;
	unless (eval { symlink("",""); 1 })
	{
		return 1;
	}
	my $retry = 30;
	while (!symlink('.', $lockFile))
	{
		if(--$retry <= 0)
		{
			return 0;
		}
		sleep(1);
	}
	return 1;
}

# Parses the parameter of CGI.
sub parseCgiVars()
{
	my $queryString = $ENV{'QUERY_STRING'};
	if ($ENV{'REQUEST_METHOD'} eq 'POST')
	{
    	$queryString .= '&' if ($queryString ne '');
	    my $content;
		read(STDIN, $content, $ENV{'CONTENT_LENGTH'});
		$queryString .= $content;
	}
	return parseUrlParam($queryString);
}

# Parses the parameter of URL.
sub parseUrlParam($)
{
	my ($queryString) = @_;
	my %result;
	foreach my $curString (split(/&/,$queryString))
	{
		my ($key, $value) = split(/=/,$curString);
		$result{decodeUrlParam($key)} = decodeUrlParam($value);
	}
	return %result;
}

# decode the parameter of URL.
sub decodeUrlParam($)
{
	$_ = @_[0];
	s/\+/ /g;
	s/%(..)/pack('c',hex($1))/ge;
	return $_;
}


#############################################################################
# Main Routine
#############################################################################

try
{
	my %cgiVars = parseCgiVars();
	my $command = $cgiVars{'c'};

	binmode(STDOUT);
	use IO::Handle;
	autoflush stdout 1;

	print "Content-Type: text/plain\nCache-Control: no-cache\n\n";

	lock($loglock) || die 'cannot lock log';
	try {
		open(LOG, ">>$logfile") || die;
		my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
		$year += 1900;
		$mon++;
		printf LOG "%4d/%02d/%02d\t%02d:%02d:%02d\t", $year, $mon, $mday, $hour, $min, $sec;
		print LOG gethostbyaddr(pack('C4', split(/\./, $ENV{'REMOTE_ADDR'})), 2)."\t";
		print LOG $ENV{'REMOTE_ADDR'}."\t";
		print LOG $ENV{'HTTP_USER_AGENT'}."\t";
		print LOG join('&', map { $_ . '=' . $cgiVars{$_} } sort keys %cgiVars)."\n";
		close(LOG);
	}
	finally {
		unlink($loglock);
	};

	if ($command =~ /^CHAL\b/)
	{
		my $challengeCode = get_current_challenge();
		response "+OK $challengeCode\n";
	}
	elsif ($command =~ /^OPEN\b/)
	{
		unless ($command =~ /^OPEN ([^\s]+) ([^\s]+)/)
		{
			response "-ERR 400 Invalid Param\n";
			exit;
		}
		my %userinfo = digest_authorize($1, $2);
		unless (%userinfo)
		{
			my $challenge = get_current_challenge();
			response "-ERR 401 $challenge Digest Authorization Required\n";
			exit;
		}
		my $sessionKey = create_session($userinfo{'username'});
		if ($sessionKey eq '')
		{
			response "-ERR 400 Already Opened\n";
			exit;
		}
		response "+OK $sessionKey\n";
	}
	elsif ($command =~ /^QUIT\b/)
	{
		verify_session($cgiVars{'k'});
		delete_session($cgiVars{'k'});
		response "+OK\n";
	}
	elsif ($command =~ /^STAT\b/)
	{
		my %userinfo = verify_session($cgiVars{'k'});
		my $filename = $userinfo{'homedir'} . $messagefile;
		open(MESSAGE, $filename) || die;
		my $linecount = 0;
		while(<MESSAGE>)
		{
			++$linecount;
		}
		close(MESSAGE);
		my $bytes = (stat($filename))[7];
		response "+OK $linecount $bytes\n";
	}
	elsif ($command =~ /^RETR\b/)
	{
		my %userinfo = verify_session($cgiVars{'k'});
		unless ($command =~ /^RETR (\d+)( (\d+))?/)
		{
			response "-ERR 400 Invalid Param\n";
			exit;
		}
		my $messageNumber = $1;
		my $messageCount = $2;
		$messageCount = 1 if ($messageCount eq '');
		if ($messageNumber <= 0 || $messageCount <= 0)
		{
			response "-ERR 400 Invalid Param\n";
			exit;
		}
		my $filename = $userinfo{'homedir'} . $messagefile;
		open(MESSAGE, $filename) || die;
		my $linenumber = 1;
		my @responseBody = "+OK\n";
		while(<MESSAGE>)
		{
			if ($linenumber >= $messageNumber + $messageCount)
			{
				last;
			}
			elsif ($linenumber >= $messageNumber)
			{
				push @responseBody, $_;
			}
			++$linenumber;
		}
		close(MESSAGE);
		push @responseBody, ".\n";
		response @responseBody;
	}
	elsif ($command =~ /^DELE\b/)
	{
		my %userinfo = verify_session($cgiVars{'k'});
		unless ($command =~ /^DELE (\d+)( (\d+))?/)
		{
			response "-ERR 400 Invalid Param\n";
			exit;
		}
		my $messageNumber = $1;
		my $messageCount = $2;
		$messageCount = 1 if ($messageCount eq '');
		if ($messageNumber <= 0 || $messageCount <= 0)
		{
			response "-ERR 400 Invalid Param\n";
			exit;
		}
		my $filename = $userinfo{'homedir'} . $messagefile;
		open(MESSAGE, $filename) || die;
		my $linenumber = 1;
		my @messages;
		while(<MESSAGE>)
		{
			if ($linenumber >= $messageNumber + $messageCount || $linenumber < $messageNumber)
			{
				push @messages, $_;
			}
			++$linenumber;
		}
		close(MESSAGE);
		open(MESSAGE, ">$filename") || die;
		foreach (@messages)
		{
			print MESSAGE $_;
		}
		close(MESSAGE);
		response "+OK\n";
	}
	elsif ($command =~ /^APND/)
	{
		unless ($command =~ /^APND ([^\s]+) ([^\s]+) (.+)/)
		{
			response "-ERR 400 Invalid Param\n";
			exit;
		}
		my $text = $3;
		my %userinfo = basic_authorize($1, $2);
		unless (%userinfo)
		{
			response "-ERR 400 Basic Authorization Failed\n";
			exit;
		}
		my $filename = $userinfo{'homedir'} . $messagefile;
		open(MESSAGE, '>>' . $filename) || die;
		print MESSAGE $text . "\n";
		close(MESSAGE);
		response "+OK\n";
	}
	elsif ($command =~ /^NOOP/)
	{
		verify_session($cgiVars{'k'});
		response "+OK\n";
	}
	else
	{
		response "-ERR 400 Invalid Command\n";
	}
}
finally {
	response "-ERR 500 Internal Server Error\n" unless $responsePrinted;
};

exit;

__END__

